/*
  ---------------------------------------------
  X11 / GdkPixbuf to GigEVision and/or Sapera info
  -----------------------------------------------
*/
#include "stdint.h"
/* 
  X11 format constants
*/

#define CORX11_DATA_FORMAT_DEFAULT  0x00000000
#define CORX11_DATA_FORMAT_MONO     0x00000001
#define CORX11_DATA_FORMAT_RGB888   0x00001000
#define CORX11_DATA_FORMAT_RGB8888  0x00001001
#define CORX11_DATA_FORMAT_RGB5551  0x00001002
#define CORX11_DATA_FORMAT_RGB565   0x00001003
#define CORX11_DATA_FORMAT_RGB101010   0x00001004

#define CORX11_DATA_FORMAT_YUV411	0x00002001
#define CORX11_DATA_FORMAT_YUV422	0x00002002
#define CORX11_DATA_FORMAT_YUV444	0x00002004

/* Sapera Format constants       */
/* Buffer data types definitions */
// (32-bit format descriptor)
//-------------------------------------------------------------------------------
// Bits     Description
//-------------------------------------------------------------------------------
// 0-7:     WiT compatible type (used by Sapera Processing 4.2 and previous)
// 8-15:    Number of bits per pixel
// 16-21:   Index (used by internal implementation as an index to function table)
// 22-23:   Planar type (specifies the type color space on planar formats)
// 24-27:   Number of pages (specifies the number of pages of planar formats)
// 28:      Sign (used by monochrome formats)
// 29:      Interlacing (used by LUT formats)
// 30:      Color (true if it's a color format)
// 31:      Not used    
//-------------------------------------------------------------------------------

#ifndef CORDATA_FORMAT_MAX
#define CORDATA_FORMAT_MAX	28
#endif

#define CORDATA_FORMAT( nPages, index, nBits)\
		(((nPages) << 24) | ((index) << 16) | (nBits << 8))

// Bit fields

// Color: bit 30
#define CORDATA_FORMAT_MONO      0x00000000
#define CORDATA_FORMAT_COLOR		0x40000000
// Interlacing: bit 29
#define CORDATA_FORMAT_NINTRL		0x00000000
#define CORDATA_FORMAT_INTRL		0x20000000
// Sign: bit 28
#define CORDATA_FORMAT_UNSIGNED	0x00000000
#define CORDATA_FORMAT_SIGNED		0x10000000
// Planar type: bits 22-23
#define CORDATA_FORMAT_PLANAR_TYPE_RGB      0x00000000   // 00
#define CORDATA_FORMAT_PLANAR_TYPE_YUV		  0x00400000   // 01
#define CORDATA_FORMAT_PLANAR_TYPE_HSV		  0x00800000   // 10
#define CORDATA_FORMAT_PLANAR_TYPE_HSI		  0x00C00000   // 11

#define CORDATA_FORMAT_COLORI		(CORDATA_FORMAT_COLOR | CORDATA_FORMAT_INTRL)
#define CORDATA_FORMAT_COLORNI	(CORDATA_FORMAT_COLOR | CORDATA_FORMAT_NINTRL)

// Monochrome data formats
#define CORDATA_FORMAT_MONO8     CORDATA_FORMAT( 1, 1, 8)
#define CORDATA_FORMAT_INT8		(CORDATA_FORMAT_MONO8 | CORDATA_FORMAT_SIGNED)
#define CORDATA_FORMAT_UINT8		(CORDATA_FORMAT_MONO8 | CORDATA_FORMAT_UNSIGNED)

#define CORDATA_FORMAT_MONO16    CORDATA_FORMAT( 1, 2, 16)
#define CORDATA_FORMAT_INT16		(CORDATA_FORMAT_MONO16 | CORDATA_FORMAT_SIGNED)
#define CORDATA_FORMAT_UINT16		(CORDATA_FORMAT_MONO16 | CORDATA_FORMAT_UNSIGNED)

#define CORDATA_FORMAT_MONO24    CORDATA_FORMAT( 1, 3, 24)
#define CORDATA_FORMAT_INT24		(CORDATA_FORMAT_MONO24 | CORDATA_FORMAT_SIGNED)
#define CORDATA_FORMAT_UINT24		(CORDATA_FORMAT_MONO24 | CORDATA_FORMAT_UNSIGNED)

#define CORDATA_FORMAT_MONO32    CORDATA_FORMAT( 1, 4, 32)
#define CORDATA_FORMAT_INT32		(CORDATA_FORMAT_MONO32 | CORDATA_FORMAT_SIGNED)
#define CORDATA_FORMAT_UINT32		(CORDATA_FORMAT_MONO32 | CORDATA_FORMAT_UNSIGNED)

#define CORDATA_FORMAT_MONO64    CORDATA_FORMAT( 1, 5, 64)
#define CORDATA_FORMAT_INT64		(CORDATA_FORMAT_MONO64 | CORDATA_FORMAT_SIGNED)
#define CORDATA_FORMAT_UINT64		(CORDATA_FORMAT_MONO64 | CORDATA_FORMAT_UNSIGNED)

// Color RGB data formats
#define CORDATA_FORMAT_RGB5551   (CORDATA_FORMAT( 1, 6, 16) | CORDATA_FORMAT_COLORI)
#define CORDATA_FORMAT_RGB565    (CORDATA_FORMAT( 1, 7, 16) | CORDATA_FORMAT_COLORI)
#define CORDATA_FORMAT_RGB888    (CORDATA_FORMAT( 1, 8, 24) | CORDATA_FORMAT_COLORI)
#define CORDATA_FORMAT_RGB8888   (CORDATA_FORMAT( 1, 9, 32) | CORDATA_FORMAT_COLORI)
#define CORDATA_FORMAT_RGB101010 (CORDATA_FORMAT( 1, 10, 32) | CORDATA_FORMAT_COLORI)
#define CORDATA_FORMAT_RGB161616 (CORDATA_FORMAT( 1, 11, 48) | CORDATA_FORMAT_COLORI)
// CORDATA_FORMAT_RGB16161616 defined below

// Color HSV data formats
#define CORDATA_FORMAT_HSV			(CORDATA_FORMAT( 1, 12, 32) | CORDATA_FORMAT_COLORI)

// Color YUV data formats
// these 4:2:2
#define CORDATA_FORMAT_UYVY		(CORDATA_FORMAT( 1, 13, 16) | CORDATA_FORMAT_COLORI)
#define CORDATA_FORMAT_YUY2		(CORDATA_FORMAT( 1, 14, 16) | CORDATA_FORMAT_COLORI)
#define CORDATA_FORMAT_YVYU		(CORDATA_FORMAT( 1, 15, 16) | CORDATA_FORMAT_COLORI)
#define CORDATA_FORMAT_YUYV		(CORDATA_FORMAT( 1, 16, 16) | CORDATA_FORMAT_COLORI)
//4:1:1 also known as Y41P
#define CORDATA_FORMAT_Y411		(CORDATA_FORMAT( 1, 17, 12) | CORDATA_FORMAT_COLORI)

//2:1:1
#define CORDATA_FORMAT_Y211		(CORDATA_FORMAT( 1, 18, 12) | CORDATA_FORMAT_COLORI)
#define CORDATA_FORMAT_YUV			(CORDATA_FORMAT( 1, 19, 32) | CORDATA_FORMAT_COLORI)
#define CORDATA_FORMAT_AYUV	   CORDATA_FORMAT_YUV

// Other data formats
#define CORDATA_FORMAT_FLOAT    	(CORDATA_FORMAT( 1, 20, 32) | CORDATA_FORMAT_SIGNED)
#define CORDATA_FORMAT_COMPLEX  	(CORDATA_FORMAT( 1, 21, 64) | CORDATA_FORMAT_SIGNED)
#define CORDATA_FORMAT_POINT	  	(CORDATA_FORMAT( 1, 22, 64) | CORDATA_FORMAT_SIGNED)
#define CORDATA_FORMAT_FPOINT	  	(CORDATA_FORMAT( 1, 23, 64) | CORDATA_FORMAT_SIGNED)

#define CORDATA_FORMAT_MONO1     (CORDATA_FORMAT( 1, 24, 1))
#define CORDATA_FORMAT_UINT1		(CORDATA_FORMAT_MONO1 | CORDATA_FORMAT_UNSIGNED)
#define CORDATA_FORMAT_BINARY	   CORDATA_FORMAT_UINT1
#define CORDATA_FORMAT_HSI			(CORDATA_FORMAT( 1, 25, 32) | CORDATA_FORMAT_COLORI)

#define CORDATA_FORMAT_IYU1      CORDATA_FORMAT_Y411
#define CORDATA_FORMAT_IYU2		(CORDATA_FORMAT( 1, 26, 24) | CORDATA_FORMAT_COLORI)
#define CORDATA_FORMAT_RGB16161616 (CORDATA_FORMAT( 1, 27, 64) | CORDATA_FORMAT_COLORI)
// add new format here!

// Planar formats (those formats reuse monochrome indexes)
#define CORDATA_FORMAT_RGBP8     (CORDATA_FORMAT( 3, 1, 8) | CORDATA_FORMAT_PLANAR_TYPE_RGB)
#define CORDATA_FORMAT_RGBP16    (CORDATA_FORMAT( 3, 2, 16) | CORDATA_FORMAT_PLANAR_TYPE_RGB)
#define CORDATA_FORMAT_YUVP8		(CORDATA_FORMAT( 3, 1, 8) | CORDATA_FORMAT_PLANAR_TYPE_YUV)
#define CORDATA_FORMAT_YUVP16		(CORDATA_FORMAT( 3, 2, 16) | CORDATA_FORMAT_PLANAR_TYPE_YUV)
#define CORDATA_FORMAT_HSVP8		(CORDATA_FORMAT( 3, 1, 8) | CORDATA_FORMAT_PLANAR_TYPE_HSV)
#define CORDATA_FORMAT_HSVP16		(CORDATA_FORMAT( 3, 2, 16) | CORDATA_FORMAT_PLANAR_TYPE_HSV)
#define CORDATA_FORMAT_HSIP8		(CORDATA_FORMAT( 3, 1, 8) | CORDATA_FORMAT_PLANAR_TYPE_HSI)
#define CORDATA_FORMAT_HSIP16		(CORDATA_FORMAT( 3, 2, 16) | CORDATA_FORMAT_PLANAR_TYPE_HSI)

// Data formats used by LUT
#define CORDATA_FORMAT_MONO9     CORDATA_FORMAT( 1, 0, 9)
#define CORDATA_FORMAT_INT9		(CORDATA_FORMAT_MONO9 | CORDATA_FORMAT_SIGNED)
#define CORDATA_FORMAT_UINT9		(CORDATA_FORMAT_MONO9 | CORDATA_FORMAT_UNSIGNED)

#define CORDATA_FORMAT_MONO10    CORDATA_FORMAT( 1, 0, 10)
#define CORDATA_FORMAT_INT10		(CORDATA_FORMAT_MONO10 | CORDATA_FORMAT_SIGNED)
#define CORDATA_FORMAT_UINT10		(CORDATA_FORMAT_MONO10 | CORDATA_FORMAT_UNSIGNED)

#define CORDATA_FORMAT_MONO11    CORDATA_FORMAT( 1, 0, 11)
#define CORDATA_FORMAT_INT11		(CORDATA_FORMAT_MONO11 | CORDATA_FORMAT_SIGNED)
#define CORDATA_FORMAT_UINT11		(CORDATA_FORMAT_MONO11 | CORDATA_FORMAT_UNSIGNED)

#define CORDATA_FORMAT_MONO12    CORDATA_FORMAT( 1, 0, 12)
#define CORDATA_FORMAT_INT12		(CORDATA_FORMAT_MONO12 | CORDATA_FORMAT_SIGNED)
#define CORDATA_FORMAT_UINT12		(CORDATA_FORMAT_MONO12 | CORDATA_FORMAT_UNSIGNED)

#define CORDATA_FORMAT_MONO13    CORDATA_FORMAT( 1, 0, 13)
#define CORDATA_FORMAT_INT13		(CORDATA_FORMAT_MONO13 | CORDATA_FORMAT_SIGNED)
#define CORDATA_FORMAT_UINT13		(CORDATA_FORMAT_MONO13 | CORDATA_FORMAT_UNSIGNED)

#define CORDATA_FORMAT_MONO14    CORDATA_FORMAT( 1, 0, 14)
#define CORDATA_FORMAT_INT14		(CORDATA_FORMAT_MONO14 | CORDATA_FORMAT_SIGNED)
#define CORDATA_FORMAT_UINT14		(CORDATA_FORMAT_MONO14 | CORDATA_FORMAT_UNSIGNED)

#define CORDATA_FORMAT_MONO15    CORDATA_FORMAT( 1, 0, 15)
#define CORDATA_FORMAT_INT15		(CORDATA_FORMAT_MONO15 | CORDATA_FORMAT_SIGNED)
#define CORDATA_FORMAT_UINT15		(CORDATA_FORMAT_MONO15 | CORDATA_FORMAT_UNSIGNED)

#define CORDATA_FORMAT_COLORI8	(CORDATA_FORMAT( 1, 0, 24) | CORDATA_FORMAT_COLORI)
#define CORDATA_FORMAT_COLORI9	(CORDATA_FORMAT( 1, 0, 27) | CORDATA_FORMAT_COLORI)
#define CORDATA_FORMAT_COLORI10	(CORDATA_FORMAT( 1, 0, 30) | CORDATA_FORMAT_COLORI)
#define CORDATA_FORMAT_COLORI11	(CORDATA_FORMAT( 1, 0, 33) | CORDATA_FORMAT_COLORI)
#define CORDATA_FORMAT_COLORI12	(CORDATA_FORMAT( 1, 0, 36) | CORDATA_FORMAT_COLORI)
#define CORDATA_FORMAT_COLORI13	(CORDATA_FORMAT( 1, 0, 39) | CORDATA_FORMAT_COLORI)
#define CORDATA_FORMAT_COLORI14	(CORDATA_FORMAT( 1, 0, 42) | CORDATA_FORMAT_COLORI)
#define CORDATA_FORMAT_COLORI15	(CORDATA_FORMAT( 1, 0, 45) | CORDATA_FORMAT_COLORI)
#define CORDATA_FORMAT_COLORI16	(CORDATA_FORMAT( 1, 0, 48) | CORDATA_FORMAT_COLORI)

#define CORDATA_FORMAT_COLORNI8	(CORDATA_FORMAT( 3, 0, 8) | CORDATA_FORMAT_COLORNI)
#define CORDATA_FORMAT_COLORNI9	(CORDATA_FORMAT( 3, 0, 9) | CORDATA_FORMAT_COLORNI)
#define CORDATA_FORMAT_COLORNI10	(CORDATA_FORMAT( 3, 0, 10) | CORDATA_FORMAT_COLORNI)
#define CORDATA_FORMAT_COLORNI11	(CORDATA_FORMAT( 3, 0, 11) | CORDATA_FORMAT_COLORNI)
#define CORDATA_FORMAT_COLORNI12	(CORDATA_FORMAT( 3, 0, 12) | CORDATA_FORMAT_COLORNI)
#define CORDATA_FORMAT_COLORNI13	(CORDATA_FORMAT( 3, 0, 13) | CORDATA_FORMAT_COLORNI)
#define CORDATA_FORMAT_COLORNI14	(CORDATA_FORMAT( 3, 0, 14) | CORDATA_FORMAT_COLORNI)
#define CORDATA_FORMAT_COLORNI15	(CORDATA_FORMAT( 3, 0, 15) | CORDATA_FORMAT_COLORNI)
#define CORDATA_FORMAT_COLORNI16	(CORDATA_FORMAT( 3, 0, 16) | CORDATA_FORMAT_COLORNI)

// Macros to access bitfield information
#define CORDATA_FORMAT_DATADEPTH(format)			(((format) >>  8) & 0xFF)
#define CORDATA_FORMAT_INDEX(format)				(((format) >> 16) & 0x3F)
#define CORDATA_FORMAT_PLANAR_TYPE(format)		(((format) >> 22) & 0x03)
#define CORDATA_FORMAT_NPAGES(format)				(((format) >> 24) & 0x0F)
#define CORDATA_FORMAT_IS_SIGNED(format)			(((format) & CORDATA_FORMAT_SIGNED)? CORDATA_FORMAT_SIGNED: CORDATA_FORMAT_UNSIGNED)
#define CORDATA_FORMAT_IS_INTRL( format)			(((format) & CORDATA_FORMAT_INTRL)? TRUE: FALSE)
#define CORDATA_FORMAT_IS_COLOR( format)			(((format) & CORDATA_FORMAT_COLOR)? TRUE: FALSE)
#define CORDATA_FORMAT_SET_SIGN( format, sign)\
{\
	if (CORDATA_FORMAT_IS_MONO(format))\
	{\
		(format) = (format) & ~CORDATA_FORMAT_SIGNED;\
		(format) = (format) | (sign);\
	}\
}

#define CORDATA_FORMAT_PLANE(format)      (CORDATA_FORMAT_NPAGES(format))
#define CORDATA_FORMAT_DATASIZE(format)	((CORDATA_FORMAT_DATADEPTH(format) + 7) >> 3)

#ifndef CORDATA_FORMAT_IS_RGB
#define CORDATA_FORMAT_IS_RGB(x) (										\
	(CORDATA_FORMAT_INDEX(x) == CORDATA_FORMAT_INDEX(CORDATA_FORMAT_RGB5551))	|| \
	(CORDATA_FORMAT_INDEX(x) == CORDATA_FORMAT_INDEX(CORDATA_FORMAT_RGB565))	|| \
	(CORDATA_FORMAT_INDEX(x) == CORDATA_FORMAT_INDEX(CORDATA_FORMAT_RGB888))	|| \
	(CORDATA_FORMAT_INDEX(x) == CORDATA_FORMAT_INDEX(CORDATA_FORMAT_RGB8888))	|| \
	(CORDATA_FORMAT_INDEX(x) == CORDATA_FORMAT_INDEX(CORDATA_FORMAT_RGB101010))|| \
	(CORDATA_FORMAT_INDEX(x) == CORDATA_FORMAT_INDEX(CORDATA_FORMAT_RGB161616)) || \
   (CORDATA_FORMAT_INDEX(x) == CORDATA_FORMAT_INDEX(CORDATA_FORMAT_RGB16161616)))
#endif

#define CORDATA_FORMAT_IS_YUV(x) (								\
	(CORDATA_FORMAT_INDEX(x) == CORDATA_FORMAT_INDEX(CORDATA_FORMAT_UYVY))	|| \
	(CORDATA_FORMAT_INDEX(x) == CORDATA_FORMAT_INDEX(CORDATA_FORMAT_YUY2))	|| \
	(CORDATA_FORMAT_INDEX(x) == CORDATA_FORMAT_INDEX(CORDATA_FORMAT_YVYU))	|| \
	(CORDATA_FORMAT_INDEX(x) == CORDATA_FORMAT_INDEX(CORDATA_FORMAT_YUYV))	|| \
	(CORDATA_FORMAT_INDEX(x) == CORDATA_FORMAT_INDEX(CORDATA_FORMAT_Y411))	|| \
	(CORDATA_FORMAT_INDEX(x) == CORDATA_FORMAT_INDEX(CORDATA_FORMAT_Y211))	|| \
	(CORDATA_FORMAT_INDEX(x) == CORDATA_FORMAT_INDEX(CORDATA_FORMAT_IYU1))	|| \
	(CORDATA_FORMAT_INDEX(x) == CORDATA_FORMAT_INDEX(CORDATA_FORMAT_IYU2))	|| \
	(CORDATA_FORMAT_INDEX(x) == CORDATA_FORMAT_INDEX(CORDATA_FORMAT_YUV)))

#ifndef CORDATA_FORMAT_IS_MONO
#define CORDATA_FORMAT_IS_MONO(x) ( \
	(CORDATA_FORMAT_INDEX( x) == CORDATA_FORMAT_INDEX( CORDATA_FORMAT_MONO1))  || \
	(CORDATA_FORMAT_INDEX( x) == CORDATA_FORMAT_INDEX( CORDATA_FORMAT_MONO8))  || \
	(CORDATA_FORMAT_INDEX( x) == CORDATA_FORMAT_INDEX( CORDATA_FORMAT_MONO16)) || \
	(CORDATA_FORMAT_INDEX( x) == CORDATA_FORMAT_INDEX( CORDATA_FORMAT_MONO24)) || \
	(CORDATA_FORMAT_INDEX( x) == CORDATA_FORMAT_INDEX( CORDATA_FORMAT_MONO32)) || \
	(CORDATA_FORMAT_INDEX( x) == CORDATA_FORMAT_INDEX( CORDATA_FORMAT_MONO64)))
#endif

/* Gray-level limits macros */
#define CORDATA_MININT8		(0)
#define CORDATA_MAXINT8		(255)
#define CORDATA_MIDINT8		(128)
#define CORDATA_NEGINT8		(-128)
#define CORDATA_POSINT8		(127)

#define CORDATA_MININT16	(0)
#define CORDATA_MAXINT16	(65535)
#define CORDATA_MIDINT16	(32768)
#define CORDATA_NEGINT16	(-32768)
#define CORDATA_POSINT16	(32767)

#define CORDATA_MININT32	(0)
#define CORDATA_MAXINT32	(4294967295)
#define CORDATA_MIDINT32	(2147483648)
#define CORDATA_NEGINT32	(-2147483647-1)
#define CORDATA_POSINT32	(2147483647)

/* Processing-specific data formats */
#define CORDATA_MIN(format) ((INT32) ( \
	format == CORDATA_FORMAT_INT8		? CORDATA_NEGINT8 : \
	format == CORDATA_FORMAT_UINT8	? CORDATA_MININT8 : \
	format == CORDATA_FORMAT_INT16	? CORDATA_NEGINT16 : \
	format == CORDATA_FORMAT_UINT16	? CORDATA_MININT16 : \
	format == CORDATA_FORMAT_INT32	? CORDATA_NEGINT32 : \
	format == CORDATA_FORMAT_UINT32	? CORDATA_MININT32 : 0))

#define CORDATA_MAX(format) ((INT32) ( \
	format == CORDATA_FORMAT_INT8	   ? CORDATA_POSINT8 : \
	format == CORDATA_FORMAT_UINT8  	? CORDATA_MAXINT8 : \
	format == CORDATA_FORMAT_INT16	? CORDATA_POSINT16 : \
	format == CORDATA_FORMAT_UINT16	? CORDATA_MAXINT16 : \
	format == CORDATA_FORMAT_INT32	? CORDATA_POSINT32 : \
	format == CORDATA_FORMAT_UINT32	? CORDATA_MAXINT32 : 0))

#define CORDATA_MID(format) ((INT32) ( \
	format == CORDATA_FORMAT_INT8		? CORDATA_MIDINT8 : \
	format == CORDATA_FORMAT_UINT8	? CORDATA_MIDINT8 : \
	format == CORDATA_FORMAT_INT16	? CORDATA_MIDINT16 : \
	format == CORDATA_FORMAT_UINT16	? CORDATA_MIDINT16 : \
	format == CORDATA_FORMAT_INT32	? CORDATA_MIDINT32 : \
	format == CORDATA_FORMAT_UINT32	? CORDATA_MIDINT32 : 0))




#ifdef __cplusplus
extern "C" {
#endif

extern int Convert_SaperaFormat_To_X11( int SaperaDataFormat);
extern int Convert_GevFormat_To_X11( int GevDataFormat);
extern int Convert_GevFormat_To_Sapera( int GevDataFormat);
extern void ConvertGevImageToX11Format( int w, int h, int gev_depth, int gev_format, void *gev_input_data, 
											int x11_depth, int x11_format, void *x11_output_data);
extern void ConvertGevImageToRGB8888Format( int w, int h, int gev_depth, int gev_format, void *gev_input_data, void *rgb_output_data);
extern void ConvertGevImageToRGB888Format( int w, int h, int gev_depth, int gev_format, void *gev_input_data, void *rgb_output_data);

// Helper functions for figuring out how to display data (with X11).

#ifndef UINT32
#define UINT32 uint32_t
#endif
extern int IsGevPixelTypeX11Displayable( int format );
extern int GetX11DisplayablePixelFormat( int convertBayer, UINT32 rawGevPixelFormat, UINT32 *convertedGevPixelFormat, UINT32 *displayableSaperaPixelFormat);

// Helper functions for reading/writing TIFF files for image archiving. (Write based on pixel format, restore to a pixel format (within reason).

extern int Read_TIFF_ToGevImage( char *filename, uint32_t *width, uint32_t *height, int pixel_format, int size, void *imageData);
extern int Write_GevImage_ToTIFF( char *filename, uint32_t width, uint32_t height, uint32_t pixel_format, void *imageData);


#ifdef __cplusplus
}
#endif 
